/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: Config_netX50.c $
   Last Modification:
    $Author: stephans $
    $Modtime: 26.02.08 17:04 $
    $Revision: 6502 $

   Targets:
    rcX          : yes

   Description:

    Configuration file for netX 50 based hardware

      * HW included:
        - UART0 + IRQ
        - TIMER0 + IRQ
        - SPI + IRQ
        - SYSLED

   Changes:

     Version  Date           Author  Description
     ----------------------------------------------------------------------------------
      1       01.03.2011     SS      initial release for netX50
**************************************************************************************/

/*
************************************************************
*   Inclusion Area
************************************************************
*/

/* Standard Includes */
#include "rX_Includes.h" 
#include "netX50regs.h"
#include "mmio_config.h"
#include "boottokens.h"
#include "AP_Urt.h"
#include "AP_LED.h"
#include "AP_SPI.h"


/*
************************************************************
*   Externals needed for patches
************************************************************
*/

/*******************************************************************************
  Local Defines - Memory size configuration
*******************************************************************************/
/* netX CPU peripherals clock rate */
#define NETX_FREQUENCY_100MHZ          100000000

#define NXHX_SPI_CHANNEL 1 /* Channel 0 on connector X5
                              Channel 1 on connector X6 */
                              
/*******************************************************************************
  Definition of the Hardware Timer Instances
*******************************************************************************/
STATIC CONST FAR RX_HWTIMER_SET_T atrXHwTim[] = {
  {
    {"SYSTIMER",RX_PERIPHERAL_TYPE_TIMER,0}, /* System Timer */
    0, /* use GPIO_counter0 */
    1000, /* 1000 microseconds = 1msec */
    TRUE, /* Continuous Mode */
    TRUE, /* Interrupt enabled */
    FALSE, /* No external Clock */
    RX_HWTIMER_TRIGGER_NONE, /* No Trigger */
    0, /* No I/O reference */
    0 /* No Prescaler */
  }
};

/*******************************************************************************
  Configuration of the SPI Instance
*******************************************************************************/
STATIC CONST FAR RX_SPISLAVE_SET_T atrXSpi[] = {
  {
    {"SERDPM", RX_PERIPHERAL_TYPE_SPI, 0}, // first SPI instance
    1,                                     // index of the slave's spi port
    0,                                     // index of the slave's chip select
    RX_SPI_MODE3,                          // spi mode for this slave device
    RX_SPI_SPEED_5_0MHz,                   // clock speed for this slave
    100,                                   // size of a burst block
    0                                      // delay between 2 burst blocks
  },
};

/*******************************************************************************
  Configuration of the UART Instance
*******************************************************************************/
STATIC CONST FAR RX_UART_SET_T atrXUrt[] = {
  {
    {"REALUART", RX_PERIPHERAL_TYPE_UART, 0},           // first "REALUART" instance
    0,                                                  // use UART 0 (first real UART)
    RX_UART_BAUDRATE_115200,                            // baud rate 115,2k
    RX_UART_PARITY_NONE,                                // no parity
    RX_UART_STOPBIT_1,                                  // 1 stop bit
    RX_UART_DATABIT_8,                                  // 8 data bits
    0,                                                  // "rx ready" trigger level for rx FIFO (set to 1..16 to enable FIFO)
    0,                                                  // "tx empty" trigger level for tx FIFO (set to 1..16 to enable FIFO)
    RX_UART_RTS_NONE,                                   // no RTS in use
    RX_UART_RTS_DEFAULT,                                // no RTS in use
    0,                                                  // no RTS forerun
    0,                                                  // no RTS trail
    RX_UART_CTS_NONE,                                   // no CTS in use
    RX_UART_CTS_DEFAULT                                 // default CTS handling
  },
};

/*******************************************************************************
  Definition of the Interrupt-Instances
*******************************************************************************/
STATIC CONST FAR RX_INTERRUPT_SET_T atrXInt[] = {
  {{"SYSTIMER",RX_PERIPHERAL_TYPE_INTERRUPT,0}, /* System Timer interrupt */
    SRT_NETX_VIC_IRQ_STAT_timer0, /* Use external Timer0 Interrupt */
    23, /* Priority 29 */
    RX_INTERRUPT_MODE_SYSTEM, /* Allow interrupt to be a thread */
    RX_INTERRUPT_EOI_AUTO, /* EOI self by RX */
    RX_INTERRUPT_TRIGGER_RISING_EDGE, /* Edge triggered */
    RX_INTERRUPT_PRIORITY_STANDARD, /* Normal Priority */
    RX_INTERRUPT_REENTRANCY_DISABLED, /* Interrupt itself is reentrant */
  },
  /* UART port 0 */
  {
    {"UART",RX_PERIPHERAL_TYPE_INTERRUPT,0}, /* Verbose UART interrupt */
    SRT_NETX_VIC_IRQ_STAT_uart0, /* Use external UART0 Interrupt */
    27, /* Priority 28 */
    RX_INTERRUPT_MODE_TASK, /* Interrupt will be handled in interrupt mode */
    RX_INTERRUPT_EOI_AUTO, /* EOI automatically by RX */
    RX_INTERRUPT_TRIGGER_RISING_EDGE, /* Edge triggered */
    RX_INTERRUPT_PRIORITY_STANDARD, /* Normal Priority */
    RX_INTERRUPT_REENTRANCY_DISABLED, /* Interrupt itself is not reentrant */
    TSK_PRIO_7,
    TSK_TOK_7,
    1024
  },
  {
    {"SPI", RX_PERIPHERAL_TYPE_INTERRUPT, 0},     // SPI
    SRT_NETX_VIC_IRQ_STAT_spi,                    // use SPI interrupt bit in the VIC IRQ status register
    25,                                           // interrupt priority
    RX_INTERRUPT_MODE_TASK,                       // create a separate task for the interrupt
    RX_INTERRUPT_EOI_AUTO,                        // EOI handled automatically
    RX_INTERRUPT_TRIGGER_RISING_EDGE,             // edge triggered
    RX_INTERRUPT_PRIORITY_STANDARD,               // normal priority
    RX_INTERRUPT_REENTRANCY_DISABLED,             // interrupt is not reentrant
    TSK_PRIO_8,                                   // interrupt task priority
    TSK_TOK_8,                                    // interrupt task token
    1024                                          // interrupt task stack size
  },
};

/*******************************************************************************
  Configuration of the LED Instances
*******************************************************************************/
STATIC CONST FAR RX_LED_SET_T atrXLed[] = {
  { // SYS yellow
    {"READY", RX_PERIPHERAL_TYPE_LED, 0},  // Peripheral Header
    {"RDYRUN", RX_PERIPHERAL_TYPE_LED, 0}, // Register Destination Definition
    {0},                                   // Optional Register to make LED Pin to output at startup
    {0},                                   // Optional Register to make LED Pin to output at startup
    {RX_LED_VALUE_TYPE_OR, 1, 0x01},       // LED Enable Register and Value
    {RX_LED_VALUE_TYPE_OR, 1, 0x01},       // LED Disable Register and Value
    NULL                                   // Set of LED functions
  },
  { // SYS green
    {"RUN", RX_PERIPHERAL_TYPE_LED, 0},    // Peripheral Header
    {"RDYRUN", RX_PERIPHERAL_TYPE_LED, 0}, // Register Destination Definition
    {0},                                   // Optional Register to make LED Pin to output at startup
    {0},                                   // Optional Register to make LED Pin to output at startup
    {RX_LED_VALUE_TYPE_OR, 2, 0x02},       // LED Enable Register and Value
    {RX_LED_VALUE_TYPE_OR, 2, 0x02},       // LED Disable Register and Value
    NULL                                   // Set of LED functions
  },
};

/*******************************************************************************
  Peripheral Configuration Summary to handed over to the RX-Kernel
*******************************************************************************/
STATIC CONST FAR RX_PERIPHERAL_CONFIG_T atrXCfg[] = 
{
  {RX_PERIPHERAL_TYPE_TIMER,atrXHwTim,MAX_CNT(atrXHwTim)},
  {RX_PERIPHERAL_TYPE_INTERRUPT,atrXInt,MAX_CNT(atrXInt)},
};

/* Function prototypes for cifx toolkit */
void ToolkitSampleTask(void* pvParam);

/*******************************************************************************
  Static Task List
*******************************************************************************/
STATIC CONST FAR RX_STATIC_TASK_T FAR atrXStaticTasks[] = 
{
  {"ToolkitExample",                      /* Set Identification */
   TSK_PRIO_36, TSK_TOK_36,               /* Set Priority,and Token ID */
   0,                                     /* Set Instance to 0 */
   0,                                     /* Pointer to Stack */
   0x1000,                                /* Size of Task Stack */
   0,                                     /* Threshold to maximum possible value */
   RX_TASK_AUTO_START,                    /* Start task automatically */
   (void*)ToolkitSampleTask,              /* Task function to schedule */
   NULL,                                  /* Function called when Task will be deleted */
   0,                                     /* Startup Parameter */
   {0,0,0,0,0,0,0,0}                      /* Reserved Region */
  },
};

/*******************************************************************************
  Peripheral Driver Configuration loaded after the Kernel has been started
*******************************************************************************/
STATIC CONST FAR RX_DRIVER_PERIPHERAL_CONFIG_T atrXDrvCfgPost[] = {
  {DrvLedInit,RX_PERIPHERAL_TYPE_LED,atrXLed,MAX_CNT(atrXLed)},
  {DrvUrtInit, RX_PERIPHERAL_TYPE_UART, atrXUrt, MAX_CNT(atrXUrt)},
  {DrvSpiInit, RX_PERIPHERAL_TYPE_SPI, atrXSpi, MAX_CNT(atrXSpi)}
};

/*******************************************************************************
  Common Parameter block for rX_SysEnterKernelExt
*******************************************************************************/
RX_ENTERKERNEL_PARAM_T trXEnterKernelParam=
{
    /* CPU clock rate */
    NETX_FREQUENCY_100MHZ,

    /* Timer interrupt task priority */
    {TSK_PRIO_DEF_RX_TIMER, 1024},

    /* Pointer to static Task-List */
    {atrXStaticTasks, MAX_CNT(atrXStaticTasks)},

    /* Pointer to rx kernel modules list */
    {0, 0},

    /* Pointer to the Peripherals-List */
    {atrXCfg, MAX_CNT(atrXCfg)},

    /* Pointer to the Post Peripherals-List / LoadDrivers included into */
    {atrXDrvCfgPost, MAX_CNT(atrXDrvCfgPost)},

    /* Pointer to optional Jump Table */
    {NULL, 0},

    /* Callback for special initialization */
    NULL,

    /* Pointer to the Middleware List */
    {0, 0},

    /* Scheduler component (if another scheduler is desired) */
    (RX_SCHEDULER_FUNCTIONS_T*)&g_tMLQueueScheduler,

    /* Cache enable flags */
    {FALSE, FALSE},
    {FALSE},
    NULL,
    {0x18000}
};

/*******************************************************************************
  Configuration of the MMIO matrix
*******************************************************************************/
STATIC CONST MMIO_CONFIG_T s_atMmioConfig[] =
{
#if NXHX_SPI_CHANNEL == 0
  {  0, MMIO_CONFIG_SPI1_MOSI,           0, 0 },
  {  1, MMIO_CONFIG_SPI1_MISO,           0, 0 },
  {  2, MMIO_CONFIG_SPI1_CLK,            0, 0 },
  {  3, MMIO_CONFIG_SPI1_CS0N,           0, 0 }, 
  {  4, MMIO_CONFIG_INPUT,               0, 0 },                                     
  {  6, MMIO_CONFIG_INPUT,               0, 0 },                                     
  {  5, MMIO_CONFIG_INPUT,               0, 0 },  
  {  7, MMIO_CONFIG_INPUT,               0, 0 },                                     
#elif NXHX_SPI_CHANNEL == 1
  {  0, MMIO_CONFIG_INPUT,               0, 0 },
  {  1, MMIO_CONFIG_INPUT,               0, 0 },
  {  2, MMIO_CONFIG_INPUT,               0, 0 },
  {  3, MMIO_CONFIG_INPUT,               0, 0 }, 
  {  4, MMIO_CONFIG_SPI1_MISO,           0, 0 },                                     
  {  6, MMIO_CONFIG_SPI1_MOSI,           0, 0 },                                     
  {  5, MMIO_CONFIG_SPI1_CLK,            0, 0 },  
  {  7, MMIO_CONFIG_SPI1_CS0N,           0, 0 },                                     
#endif
  {  8, MMIO_CONFIG_GPIO8,               0, 0 },                    
  {  9, MMIO_CONFIG_GPIO9,               0, 0 },                
  { 10, MMIO_CONFIG_GPIO10,              0, 0 },                  
  { 11, MMIO_CONFIG_GPIO11,              0, 0 }, 
  { 12, MMIO_CONFIG_INPUT,               0, 0 },
  { 13, MMIO_CONFIG_INPUT,               0, 0 },
  { 14, MMIO_CONFIG_INPUT,               0, 0 },
  { 15, MMIO_CONFIG_INPUT,               0, 0 },
  { 16, MMIO_CONFIG_UART1_RXD,           0, 0 },                                     
  { 17, MMIO_CONFIG_UART1_TXD,           0, 0 },                                     
  { 18, MMIO_CONFIG_UART1_CTS,           0, 0 },                                     
  { 19, MMIO_CONFIG_UART1_RTS,           0, 0 },                                     
  { 20, MMIO_CONFIG_GPIO12,              0, 0 },                
  { 21, MMIO_CONFIG_GPIO13,              0, 0 },                                        
  { 22, MMIO_CONFIG_GPIO14,              0, 0 },                                                               
  { 23, MMIO_CONFIG_GPIO15,              0, 0 },
  { 24, MMIO_CONFIG_GPIO24,              0, 0 },   
  { 25, MMIO_CONFIG_GPIO25,              0, 0 },                          
  { 26, MMIO_CONFIG_GPIO26,              0, 0 },                               
  { 27, MMIO_CONFIG_GPIO27,              0, 0 },                          
  { 28, MMIO_CONFIG_GPIO0,               0, 0 },                   
  { 30, MMIO_CONFIG_INPUT,               0, 0 },      
#if NXHX_SPI_CHANNEL == 0
  { 29, MMIO_CONFIG_INPUT,               0, 0 },                       
  { 31, MMIO_CONFIG_SPI1_CS0N,           0, 0 },                        
#elif NXHX_SPI_CHANNEL == 1
  { 29, MMIO_CONFIG_SPI1_CS0N,           0, 0 },                       
  { 31, MMIO_CONFIG_INPUT,               0, 0 },                        
#endif
  { 32, MMIO_CONFIG_INPUT,               0, 0 },                                     
  { 33, MMIO_CONFIG_XM1_IO0,             0, 0 },                                     
  { 34, MMIO_CONFIG_UART0_RXD,           0, 0 },                                     
  { 35, MMIO_CONFIG_UART0_TXD,           0, 0 },                                     
  { 36, MMIO_CONFIG_INPUT,               0, 0 },
  { 37, MMIO_CONFIG_GPIO1,               0, 0 },
  { 38, MMIO_CONFIG_GPIO2,               0, 0 },
  { 39, MMIO_CONFIG_GPIO3,               0, 0 },
};
 
/*  =====================================================================================

  Function: main

            - Enters the RX-kernel mode forever

  ---------------------------------------------------------------------------------------

  Input   : -

  Output  : -

  Return  : -


  ======================================================================================= */
INT main (void)
{
  volatile RX_FATAL erXFat; /* Fatal Error value, in case function would return */
  volatile unsigned long* pulNetxSta = (volatile unsigned long*)NETX_SYS_STAT;
  int iIdx = 0;
   
  #define BOOTOPTION_2NDSTAGELOADER   7
  
  
  /* Check if we came from second stage loader and the system status register contains 0xAA as
     netX flags, which means we came from a cold start */
  if(rX_DevGetBoottype() == BOOTOPTION_2NDSTAGELOADER)
  {
    if((*pulNetxSta & 0x0000FF00) == 0x0000AA00)
    {
      typedef void(*PFN_RESET)(void);
      PFN_RESET pfnReset = NULL;
      
      /* we were started during coldstart and have a 2nd stage loader, so jump back to rom loader
         to allow loading of a new firmware */
      *pulNetxSta = 0;
      pfnReset();      
    }
  }   
      
  /* Configure MMIO pin multiplexer */
  for(iIdx = 0; iIdx < sizeof(s_atMmioConfig) / sizeof(s_atMmioConfig[0]); iIdx++)
  {
    volatile unsigned long* pulMmioConfig = (volatile unsigned long*)NETX_MMIO_CTRL_AREA;
    volatile unsigned long* pulAccessKey  = (volatile unsigned long*)NETX_IO_CFG_ACCESS_KEY;
    
    // Unlock MMIO Config register
    *pulAccessKey = *pulAccessKey;
    
    pulMmioConfig[s_atMmioConfig[iIdx].bMmio] = s_atMmioConfig[iIdx].bFunction << SRT_NETX_MMIO0_CFG_mmio0_sel             |
                                                ((s_atMmioConfig[iIdx].fInvertIn)  ? MSK_NETX_MMIO0_CFG_mmio0_in_inv  : 0) |
                                                ((s_atMmioConfig[iIdx].fInvertOut) ? MSK_NETX_MMIO0_CFG_mmio0_out_inv : 0);
  }
         
  /* Boot the Kernel by using the set Call-Pointer and hand over the static Task-List and the Peripherals-List */
  erXFat = rX_SysEnterKernelExt(&trXEnterKernelParam);

  /* We should never return to here */  
  while(erXFat!=RX_OK);
  
  return(0); /* Prevent Compiler warning because of non-void function */
}
